<?php
/**
 * Plugin Name: Gizmo
 * Description: Adds a REST API to import products into WooCommerce using a gizmo_id and token-based auth. Also provides an admin page to generate token and enable automatic Gizmo categories.
 * Version: 1.0.3
 * Author: Milad Rezaeian (milad.rezaeiann@gmail.com)
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Gizmo_Plugin {
    const OPTION_TOKEN = 'gizmo_token';
    const OPTION_CREATE_CATS = 'gizmo_create_categories';

    public function __construct() {
        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );
        add_action( 'admin_init', array( $this, 'admin_init' ) );
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }

    public function activate() {
        if ( get_option( self::OPTION_TOKEN ) === false ) {
            update_option( self::OPTION_TOKEN, wp_generate_password( 40, false, false ) );
        }
        if ( get_option( self::OPTION_CREATE_CATS ) === false ) {
            update_option( self::OPTION_CREATE_CATS, 0 );
        }
    }

    public function admin_menu() {
        add_menu_page(
            'Gizmo',
            'گیزمو',
            'manage_options',
            'gizmo-settings',
            array( $this, 'settings_page' ),
            plugin_dir_url( __FILE__ ) . 'assets/icon.png',
            56
        );
    }

    public function admin_init() {
//        register_setting( 'gizmo_options', self::OPTION_TOKEN );
        register_setting(
            'gizmo_options',
            self::OPTION_CREATE_CATS,
            array(
                'sanitize_callback' => array( $this, 'sanitize_create_categories' )
            )
        );
    }

    /**
     * Sanitize the gizmo_create_categories option to ensure it's a valid boolean-like value.
     *
     * @param mixed $value The value to sanitize.
     * @return int 1 if the checkbox is checked, 0 otherwise.
     */
    public function sanitize_create_categories( $value ) {
        return $value === '1' ? 1 : 0;
    }

    public function settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( isset( $_POST['gizmo_regen_token'] ) && check_admin_referer( 'gizmo_regen_token_action', 'gizmo_regen_token_nonce' ) ) {
            $new = wp_generate_password( 40, false, false );
            update_option( self::OPTION_TOKEN, $new );
            echo '<div class="updated notice"><p>✅ توکن جدید ساخته شد.</p></div>';
        }

        $token = get_option( self::OPTION_TOKEN );
        $create_cats = get_option( self::OPTION_CREATE_CATS );
        ?>
        <div class="wrap">
            <h1>تنظیمات گیزمو</h1>

            <hr>
            <h3>🔗 تنظیمات اتصال ووکامرس به گیزمو</h3>
            <p>
                برای اتصال فروشگاه ووکامرس خود به گیزمو، کافیست توکن و آدرس سایت خود را در حساب گیزمو وارد کنید.
            </p><p>
                بعد از انجام این مرحله، ارتباط بین فروشگاه شما و گیزمو برقرار می‌شود و می‌توانید محصولات را به‌صورت خودکار به ووکامرس ایمپورت کنید.
            </p>
            <p style="margin-top: 20px;">
                <a href="https://gizmods.com/blogs/woocommerce-plugin" target="_blank" class="button button-primary" style="background-color:#0073aa;border-color:#006799;">
                    🔗 آموزش اتصال ووکامرس به گیزمو
                </a>
            </p>
            <hr>

            <table class="form-table">
                <tr valign="top">
                    <th scope="row">توکن</th>
                    <td style="display: inline-flex; align-items: center; gap: 10px;">
                        <input type="text" id="gizmo_token" readonly class="regular-text" value="<?php echo esc_attr( $token ); ?>" style="direction:ltr;" />

                        <form method="post" style="margin: 0;">
                            <?php wp_nonce_field( 'gizmo_regen_token_action', 'gizmo_regen_token_nonce' ); ?>
                            <input type="submit" name="gizmo_regen_token" class="button button-secondary" value="🔄 بازسازی توکن" />
                        </form>

                        <button type="button" id="gizmo_copy_token" class="button button-secondary">📋 کپی توکن</button>
                        <span id="gizmo_copy_message" style="color:green; display:none;">✅ کپی شد!</span>
                    </td>

                </tr>
            </table>

            <hr>
            <script>
                document.addEventListener('DOMContentLoaded', function() {
                    const copyBtn = document.getElementById('gizmo_copy_token');
                    const tokenInput = document.getElementById('gizmo_token');
                    const message = document.getElementById('gizmo_copy_message');

                    if (copyBtn && tokenInput && message) {
                        copyBtn.addEventListener('click', function() {
                            tokenInput.select();
                            tokenInput.setSelectionRange(0, 99999);
                            navigator.clipboard.writeText(tokenInput.value)
                                .then(() => {
                                    message.style.display = 'inline';
                                    setTimeout(() => {
                                        message.style.display = 'none';
                                    }, 1500);
                                })
                                .catch(() => {
                                    alert('❌ Copy failed. Please copy manually.');
                                });
                        });
                    }
                });
            </script>

            <form method="post" action="options.php">
                <?php settings_fields( 'gizmo_options' ); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">ایجاد دسته‌بندی‌های گیزمو</th>
                        <td>
                            <label>
                                <input type="checkbox" name="gizmo_create_categories" value="1" <?php checked( 1, $create_cats ); ?> />
                                ایجاد دسته‌بندی‌های گیزمو
                            </label>
                            <p class="description">اگر تیک زده شود، در هنگام ایمپورت دسته‌ها ایجاد خواهند شد.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button( 'ذخیره تنظیمات' ); ?>
            </form>
        </div>


        <?php
    }



    public function register_routes() {
        register_rest_route( 'gizmo/v1', '/product', array(
            'methods' => 'POST',
            'callback' => array( $this, 'handle_product' ),
            'permission_callback' => '__return_true', // We'll check token inside callback
        ) );

        register_rest_route( 'gizmo/v1', '/update-stock-price', array(
            'methods' => 'POST',
            'callback' => array( $this, 'handle_stock_price_update' ),
            'permission_callback' => '__return_true',
        ) );

        register_rest_route( 'gizmo/v1', '/verify-token', array(
            'methods' => 'POST',
            'callback' => array( $this, 'verify_token' ),
            'permission_callback' => '__return_true',
        ) );
    }

    private function check_token( $provided ) {
        $stored = get_option( self::OPTION_TOKEN );
        return hash_equals( (string) $stored, (string) $provided );
    }

    public function handle_product( WP_REST_Request $request ) {
        $params = $request->get_json_params();
        if ( empty( $params ) ) {
            return new WP_REST_Response( array( 'error' => 'بدون داده' ), 400 );
        }

        // token can be provided as header X-Gizmo-Token or in body token
        $provided = $request->get_header( 'x-gizmo-token' );
        if ( empty( $provided ) && isset( $params['token'] ) ) {
            $provided = $params['token'];
        }

        if ( empty( $provided ) || ! $this->check_token( $provided ) ) {
            return new WP_REST_Response( array( 'error' => 'توکن نامعتبر' ), 401 );
        }

        $items = isset( $params['products'] ) && is_array( $params['products'] ) ? $params['products'] : array( $params );
        $results = array();

        foreach ( $items as $item ) {
            $r = $this->import_single_product( $item );
            $results[] = $r;
        }

        return new WP_REST_Response( array( 'results' => $results ), 200 );
    }

    private function import_single_product( $data ) {
        if ( empty( $data['title'] ) && empty( $data['name'] ) ) {
            return array( 'status' => 'error', 'message' => 'نام محصول موجود نیست', 'incoming' => $data );
        }

        $title = isset( $data['title'] ) ? sanitize_text_field( $data['title'] ) : sanitize_text_field( $data['name'] );
        $gizmo_id = isset( $data['gizmo_id'] ) ? sanitize_text_field( $data['gizmo_id'] ) : null;

        // try find existing by gizmo_id meta
        $existing_id = null;
        if ( $gizmo_id ) {
            $found = get_posts( array(
                'post_type' => 'product',
                'meta_query' => array(
                    array( 'key' => '_gizmo_id', 'value' => $gizmo_id, 'compare' => '=' )
                ),
                'posts_per_page' => 1,
            ) );
            if ( ! empty( $found ) ) {
                $existing_id = $found[0]->ID;
            }
        }

        $post_data = array(
            'post_title'   => wp_strip_all_tags( $title ),
            'post_content' => isset( $data['description'] ) ? wp_kses_post( $data['description'] ) : '',
            'post_status'  => 'publish',
            'post_type'    => 'product',
        );

        if ( $existing_id ) {
            $post_data['ID'] = $existing_id;
            $product_id = wp_update_post( $post_data );
            $action = 'updated';
        } else {
            $product_id = wp_insert_post( $post_data );
            $action = 'created';
        }

        if ( is_wp_error( $product_id ) || ! $product_id ) {
            return array( 'status' => 'error', 'message' => 'خطا در ایجاد/به‌روزرسانی محصول', 'wp_error' => $product_id );
        }

        // save gizmo_id
        if ( $gizmo_id ) {
            update_post_meta( $product_id, '_gizmo_id', $gizmo_id );
        }

        // price
        if ( isset( $data['price'] ) ) {
            $raw_price = wc_format_decimal( $data['price'] );

            $currency = get_option( 'woocommerce_currency' );

            if ( strtolower( $currency ) === 'irt' || strtolower( $currency ) === 'toman' ) {
                $raw_price = $raw_price / 10;
            }

            update_post_meta( $product_id, '_regular_price', $raw_price );
            update_post_meta( $product_id, '_price', $raw_price );
        }


        // SKU
        if ( isset( $data['sku'] ) ) {
            update_post_meta( $product_id, '_sku', sanitize_text_field( $data['sku'] ) );
        }

        // categories
        if ( isset( $data['categories'] ) && is_array( $data['categories'] ) ) {
            $term_ids = array();
            foreach ( $data['categories'] as $cat_name ) {
                $cat_name = sanitize_text_field( $cat_name );
                $term = term_exists( $cat_name, 'product_cat' );
                if ( $term === 0 || $term === null ) {
                    // only create if setting allows
                    if ( get_option( self::OPTION_CREATE_CATS ) ) {
                        $new = wp_insert_term( $cat_name, 'product_cat' );
                        if ( ! is_wp_error( $new ) && isset( $new['term_id'] ) ) {
                            $term_ids[] = intval( $new['term_id'] );
                        }
                    }
                } else {
                    $term_ids[] = is_array( $term ) ? intval( $term['term_id'] ) : intval( $term );
                }
            }
            if ( ! empty( $term_ids ) ) {
                wp_set_object_terms( $product_id, $term_ids, 'product_cat' );
            }
        }

        if ( isset( $data['attributes'] ) && is_array( $data['attributes'] ) ) {
            global $wpdb;
            $product_attributes = array();

            foreach ( $data['attributes'] as $attr ) {
                if ( empty( $attr['name'] ) || empty( $attr['values'] ) ) {
                    continue;
                }

                $name   = sanitize_text_field( $attr['name'] );
                $values = array_map( 'sanitize_text_field', (array) $attr['values'] );

                $slug = wc_sanitize_taxonomy_name( $name );
                $taxonomy = 'pa_' . $slug;

                $attribute_exists = false;
                $attributes = wc_get_attribute_taxonomies();
                foreach ( $attributes as $attribute ) {
                    if ( $attribute->attribute_name === $slug ) {
                        $attribute_exists = true;
                        break;
                    }
                }

                if ( ! $attribute_exists ) {
                    // ایجاد ویژگی جدید با استفاده از API ووکامرس
                    $attribute_id = wc_create_attribute( array(
                        'name' => $name,
                        'slug' => $slug,
                        'type' => 'select',
                        'order_by' => 'menu_order',
                        'has_archives' => false,
                    ) );

                    if ( is_wp_error( $attribute_id ) ) {
                        continue; // در صورت خطا، ادامه می‌دهیم
                    }
                }

                if ( ! taxonomy_exists( $taxonomy ) ) {
                    register_taxonomy(
                        $taxonomy,
                        array( 'product' ),
                        array(
                            'label' => $name,
                            'public' => false,
                            'show_ui' => false,
                            'hierarchical' => false,
                        )
                    );
                }

                $term_ids = array();
                foreach ( $values as $val ) {
                    $term = term_exists( $val, $taxonomy );
                    if ( ! $term ) {
                        $term = wp_insert_term( $val, $taxonomy );
                    }
                    if ( ! is_wp_error( $term ) ) {
                        $term_ids[] = intval( $term['term_id'] );
                    }
                }

                if ( ! empty( $term_ids ) ) {
                    wp_set_object_terms( $product_id, $term_ids, $taxonomy );
                }

                $product_attributes[ $taxonomy ] = array(
                    'name'         => $taxonomy,
                    'value'        => '',
                    'position'     => 0,
                    'is_visible'   => 1,
                    'is_variation' => 0,
                    'is_taxonomy'  => 1,
                );
            }

            if ( ! empty( $product_attributes ) ) {
                update_post_meta( $product_id, '_product_attributes', $product_attributes );
            }
        }

        // stock
        if ( isset( $data['stock'] ) ) {
            update_post_meta( $product_id, '_stock', intval( $data['stock'] ) );
            update_post_meta( $product_id, '_manage_stock', 'yes' );
            update_post_meta( $product_id, '_stock_status', intval( $data['stock'] ) > 0 ? 'instock' : 'outofstock' );
        }

        // short description
        if ( isset( $data['short_description'] ) ) {
            wp_update_post( array( 'ID' => $product_id, 'post_excerpt' => wp_kses_post( $data['short_description'] ) ) );
        }

        // images + feature_image
        if ( isset( $data['feature_image'] ) || ( isset( $data['images'] ) && is_array( $data['images'] ) ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';

            $attachment_ids = array();
            $featured_id = 0;

            if ( ! empty( $data['feature_image'] ) ) {
                $feature_url = esc_url_raw( $data['feature_image'] );

                $image_id = $this->gizmo_upload_image($feature_url, $product_id);
                if (is_wp_error($image_id)) {
                    return $image_id;
                }
                set_post_thumbnail($product_id, $image_id);

            }

            if ( isset( $data['images'] ) && is_array( $data['images'] ) ) {
                foreach ( $data['images'] as $img_url ) {
                    $img_url = esc_url_raw( $img_url );

                    $image_id = $this->gizmo_upload_image($img_url, $product_id);
                    if (is_wp_error($image_id)) {
                        return $image_id;
                    }

                    if ( ! is_wp_error( $image_id ) ) $attachment_ids[] = $image_id;
                }
            }

            if ( count( $attachment_ids ) > 1 ) {
                update_post_meta( $product_id, '_product_image_gallery', implode( ',', $attachment_ids ) );
            }
        }

        return array( 'status' => 'ok', 'action' => $action, 'product_id' => $product_id );
    }

    private function gizmo_upload_image($image_url, $post_id) {
        // Get the file from the URL
        $response = wp_remote_get($image_url);
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return new WP_Error('image_fetch_failed', 'Failed to fetch image from the URL', array('status' => 400));
        }

        $image_data = wp_remote_retrieve_body($response);
        $filename = basename($image_url);

        $file_array = array(
            'name' => $filename,
            'tmp_name' => tempnam(sys_get_temp_dir(), $filename),
        );


        file_put_contents($file_array['tmp_name'], $image_data);


        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');


        $attachment_id = media_handle_sideload($file_array, $post_id);
        if (is_wp_error($attachment_id)) {
            return $attachment_id;
        }

        return $attachment_id;
    }

    public function handle_stock_price_update( WP_REST_Request $request ) {
        $params = $request->get_json_params();
        if ( empty( $params ) ) {
            return new WP_REST_Response( array( 'error' => 'بدون داده' ), 400 );
        }

        $provided = $request->get_header( 'x-gizmo-token' );
        if ( empty( $provided ) && isset( $params['token'] ) ) {
            $provided = $params['token'];
        }

        if ( empty( $provided ) || ! $this->check_token( $provided ) ) {
            return new WP_REST_Response( array( 'error' => 'توکن نامعتبر' ), 401 );
        }

        $items = isset( $params['products'] ) && is_array( $params['products'] ) ? $params['products'] : array( $params );
        $results = array();

        foreach ( $items as $item ) {
            if ( empty( $item['gizmo_id'] ) ) {
                $results[] = array( 'status' => 'error', 'message' => 'gizmo_id موجود نیست' );
                continue;
            }

            $gizmo_id = sanitize_text_field( $item['gizmo_id'] );

            $found = get_posts( array(
                'post_type' => 'product',
                'meta_query' => array(
                    array( 'key' => '_gizmo_id', 'value' => $gizmo_id, 'compare' => '=' )
                ),
                'posts_per_page' => 1,
            ) );

            if ( empty( $found ) ) {
                $results[] = array( 'status' => 'error', 'gizmo_id' => $gizmo_id, 'message' => 'محصول یافت نشد' );
                continue;
            }

            $product_id = $found[0]->ID;

            // 🔹 آپدیت قیمت
//            if ( isset( $item['price'] ) ) {
//                $price = wc_format_decimal( $item['price'] );
//                update_post_meta( $product_id, '_regular_price', $price );
//                update_post_meta( $product_id, '_price', $price );
//            }
            if ( isset( $item['price'] ) ) {
                $raw_price = wc_format_decimal( $item['price'] );

                $currency = get_option( 'woocommerce_currency' );

                if ( strtolower( $currency ) === 'irt' || strtolower( $currency ) === 'toman' ) {
                    $raw_price = $raw_price / 10;
                }

                update_post_meta( $product_id, '_regular_price', $raw_price );
                update_post_meta( $product_id, '_price', $raw_price );
            }

            // 🔹 آپدیت موجودی
            if ( isset( $item['stock'] ) ) {
                $stock = intval( $item['stock'] );
                update_post_meta( $product_id, '_stock', $stock );
                update_post_meta( $product_id, '_manage_stock', 'yes' );
                update_post_meta( $product_id, '_stock_status', $stock > 0 ? 'instock' : 'outofstock' );
            }

            $results[] = array( 'status' => 'ok', 'gizmo_id' => $gizmo_id, 'product_id' => $product_id );
        }

        return new WP_REST_Response( array( 'results' => $results ), 200 );
    }

    public function verify_token( WP_REST_Request $request ) {
        $params = $request->get_json_params();
        $provided = isset( $params['token'] ) ? sanitize_text_field( $params['token'] ) : '';

        if ( empty( $provided ) ) {
            return new WP_REST_Response( array(
                'valid' => false,
                'message' => 'Token not provided.'
            ), 400 );
        }

        if ( $this->check_token( $provided ) ) {
            return new WP_REST_Response( array(
                'valid' => true,
                'message' => 'Token is valid.'
            ), 200 );
        }

        return new WP_REST_Response( array(
            'valid' => false,
            'message' => 'Invalid token.'
        ), 401 );
    }

}

new Gizmo_Plugin();


register_activation_hook( __FILE__, 'gizmo_plugin_activate' );

function gizmo_plugin_activate() {
    if ( get_option( Gizmo_Plugin::OPTION_TOKEN ) === false ) {
        update_option( Gizmo_Plugin::OPTION_TOKEN, wp_generate_password( 40, false, false ) );
    }
    if ( get_option( Gizmo_Plugin::OPTION_CREATE_CATS ) === false ) {
        update_option( Gizmo_Plugin::OPTION_CREATE_CATS, 0 );
    }
}